<?php

namespace App\Filament\Resources;

use App\Filament\Resources\AdResource\Pages;
use App\Filament\Resources\AdResource\RelationManagers;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables\Actions\Action;
use App\Models\Ad;

use Closure;
use Filament\Forms\Get;

use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Forms\Components\FileUpload;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use Illuminate\Support\Str;
use App\Settings\CardDefaultsSetting;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Filters\Filter;

class AdResource extends Resource
{
    protected static ?string $model = Ad::class;

    protected static ?string $navigationIcon = 'heroicon-o-megaphone';

    protected static ?string $navigationGroup = 'Ads Management';
    protected static ?string $navigationLabel = 'Admin Ads';
    protected static ?string $modelLabel = 'Admin Ads';
    protected static ?int $navigationSort = 6;

    public static function form(Form $form): Form
    {
        $settings = app(CardDefaultsSetting::class);
        $width = $settings->ad_banner_w;
        $height = $settings->ad_banner_h;
        $aspectRatio = "{$width}:{$height}";

        return $form->schema([

            FileUpload::make('image')
                ->label("Upload Ad Banner ({$width}x{$height})")
                ->placeholder('Upload Ad Banner')
                ->directory('ads')
                ->disk('public')
                ->image()
                ->required()
                ->imageEditor()
                ->imageCropAspectRatio($aspectRatio) // This sets the default crop shape
                ->imageEditorAspectRatios([$aspectRatio]) // Only allow this one crop shape
                ->imageResizeMode('cover') // Resize after crop
                ->imageResizeTargetWidth($width)
                ->imageResizeTargetHeight($height)
                ->columnSpanFull()
                ->getUploadedFileNameForStorageUsing(
                    fn(TemporaryUploadedFile $file): string =>
                    (string) Str::random(40) . '.' . $file->getClientOriginalExtension()
                ),

            Forms\Components\TextInput::make('caption')->label('Ad Title')
                ->required(),
            Forms\Components\TextInput::make('link')->required(),
            Forms\Components\Hidden::make('display_in')
                ->default('Free'),
            // ->reactive(),
            // Forms\Components\Select::make('sponsor_id')
            //     ->label('Sponsor Name')
            //     ->relationship('sponsor', 'sponsor_name')
            //     ->searchable()
            //     ->preload()
            //     ->reactive()
            //     ->disabled(fn(callable $get) => $get('display_in') !== 'Sponsored')
            //     ->rules([
            //         fn(Get $get): Closure => function (string $attribute, $value, Closure $fail) use ($get) {
            //             if ($get('display_in') === 'Sponsored' && empty($value)) {
            //                 $fail("Sponsor is required when creating ads for Sponsored Cards.");
            //             }
            //         },
            //     ])
            //     ->required(fn(Get $get) => $get('display_in') === 'Sponsored')
            //     ->validationMessages([
            //         'required' => 'Sponsor is required when creating ads for Sponsored Cards.',
            //     ]),
            Forms\Components\Select::make('position')
                ->columnSpanFull()
                ->required()
                ->label('Position')
                ->options([
                    'top' => 'Top',
                    'bottom' => 'Bottom',
                ]),
            Forms\Components\Toggle::make('is_active')
                ->label('Active Ad')
                ->default(true)
                ->required(),
        ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->reorderable('order')
            ->reorderRecordsTriggerAction(
                fn(Action $action, bool $isReordering) => $action
                    ->button()
                    ->label($isReordering ? 'Disable reordering' : 'Enable reordering'),
            )
            ->description('Ads posted here will be displayed on Free cards.')
            ->columns([
                Tables\Columns\ToggleColumn::make('is_active')
                    ->label('Active')
                    ->onColor('success') // ✅ green when active
                    ->offColor('danger'), // ❌ red when inactive

                Tables\Columns\TextColumn::make('caption')
                    ->label('Ad Title'),
                Tables\Columns\TextColumn::make('display_in')
                    ->formatStateUsing(fn($state) => ucfirst($state))
                    ->badge()
                    ->label('Card')
                    ->color(fn(string $state): string => match ($state) {
                        'Free' => 'info',
                        'Sponsored' => 'danger',
                        'Influencer' => 'influencer',
                    }),
                // Tables\Columns\TextColumn::make('sponsor.sponsor_name')
                //     ->label('Sponsor Name')
                //     ->searchable(),
                Tables\Columns\TextColumn::make('position')
                    ->formatStateUsing(fn($state) => ucfirst($state)),
            ])
            ->filters([
                Filter::make('is_active')
                    ->label('Active')
                    ->query(fn(Builder $query) => $query->where('is_active', true)),
                SelectFilter::make('display_in')
                    ->label('Card')
                    ->options([
                        'Free' => 'Free',
                        'Sponsored' => 'Sponsored',
                        'Influencer' => 'Influencer',
                    ]),
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getEloquentQuery(): Builder
    {
        // return parent::getEloquentQuery()->where('is_admin', true)->orderBy('order');
        return parent::getEloquentQuery()
            ->where('is_admin', true)
            ->where('display_in', 'Free')
            ->orderBy('order');
    }

    // protected static function booted()
    // {
    //     static::creating(function ($ad) {
    //         if (auth()->user()->is_admin) {
    //             $ad->is_admin = true;
    //         }
    //     });
    // }

    public static function getWidgets(): array
    {
        return [
            AdResource\Widgets\AdminAdAnalytics::class,
            AdResource\Widgets\AdminAdAnalyticsOneWeek::class,
            AdResource\Widgets\SpecificAdminAdAnalytics::class,
            AdResource\Widgets\SpecificAdminAdAnalyticsOneWeek::class,
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListAds::route('/'),
            'create' => Pages\CreateAd::route('/create'),
            'view' => Pages\ViewAd::route('/{record}'),
            'edit' => Pages\EditAd::route('/{record}/edit'),
        ];
    }
}
