<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\User;
use Laravel\Cashier\Subscription;
use Laravel\Cashier\SubscriptionItem;
use Stripe\Stripe;
use Stripe\Subscription as StripeSubscription;

class SyncStripeSubscriptions extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'stripe:sync-subscriptions {--user_id=}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Sync subscriptions and subscription items from Stripe using stripe_id in users table.';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🔄 Starting Stripe subscription sync...');

        Stripe::setApiKey(config('services.stripe.secret'));

        $query = User::whereNotNull('stripe_id');

        if ($userId = $this->option('user_id')) {
            $query->where('id', $userId);
            $this->info("Filtering by user_id={$userId}");
        }

        $totalSynced = 0;

        $query->chunk(50, function ($users) use (&$totalSynced) {
            foreach ($users as $user) {
                try {
                    $this->line("→ Syncing user [ID: {$user->id}] ({$user->email})");

                    $stripeSubs = StripeSubscription::all([
                        'customer' => $user->stripe_id,
                        'limit' => 100,
                    ]);

                    foreach ($stripeSubs->autoPagingIterator() as $stripeSub) {
                        $localSub = Subscription::updateOrCreate(
                            [
                                'stripe_id' => $stripeSub->id,
                            ],
                            [
                                'user_id' => $user->id,
                                'type' => 'default',
                                'stripe_status' => $stripeSub->status,
                                'stripe_price' => $stripeSub->items->data[0]->price->id ?? null,
                                'quantity' => $stripeSub->items->data[0]->quantity ?? 1,
                                'trial_ends_at' => $stripeSub->trial_end ? now()->setTimestamp($stripeSub->trial_end) : null,
                                'ends_at' => $stripeSub->cancel_at ? now()->setTimestamp($stripeSub->cancel_at) : null,
                            ]
                        );

                        foreach ($stripeSub->items->data as $item) {
                            SubscriptionItem::updateOrCreate(
                                [
                                    'stripe_id' => $item->id,
                                ],
                                [
                                    'subscription_id' => $localSub->id,
                                    'stripe_product' => $item->price->product,
                                    'stripe_price' => $item->price->id,
                                    'quantity' => $item->quantity,
                                ]
                            );
                        }

                        $totalSynced++;
                    }
                } catch (\Exception $e) {
                    $this->error("❌ Failed for user ID {$user->id}: " . $e->getMessage());
                    \Log::error("Stripe sync failed for user {$user->id}: " . $e->getMessage());
                }
            }
        });

        $this->info("✅ Done! Synced {$totalSynced} subscription(s).");
    }
}
