<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\User;
use App\Models\Referral;
use App\Models\Commissions;
use Laravel\Cashier\Invoice;
use Illuminate\Support\Facades\Log;
use Stripe\Stripe;
use Stripe\Invoice as StripeInvoice;

class SyncPastStripeCommissions extends Command
{
    protected $signature = 'commissions:sync-missing';
    protected $description = 'Sync missing commissions from past paid invoices';

    public function handle()
    {
        Stripe::setApiKey(config('cashier.secret'));

        $this->info('Fetching past paid invoices from Stripe...');

        $startingAfter = null;
        $createdCount = 0;

        do {
            $params = [
                'limit' => 100,
                'status' => 'paid',
            ];

            if ($startingAfter) {
                $params['starting_after'] = $startingAfter;
            }

            $invoices = \Stripe\Invoice::all($params);
            foreach ($invoices->data as $invoice) {
                $startingAfter = $invoice->id;

                // Skip if already processed
                if (Commissions::where('stripe_invoice_id', $invoice->id)->exists()) {
                    continue;
                }

                $user = User::where('stripe_id', $invoice->customer)->first();

                if (!$user) {
                    Log::warning('No user found for customer ID: ' . $invoice->customer);
                    continue;
                }

                $refer = Referral::where('referred_id', $user->id)->first();
                if (!$refer || !$refer->referrer_id_main) {
                    continue;
                }

                $referrer = User::find($refer->referrer_id_main);
                if (!$referrer) {
                    continue;
                }

                $rate = match ($referrer->partnership_level) {
                    'senior' => 0.50,
                    'junior' => 0.25,
                    'leader' => 0.20,
                    'default' => 0.10,
                    default => null,
                };

                if (!$rate) {
                    continue;
                }

                $total = $invoice->amount_paid / 100;

                Commissions::create([
                    'user_id' => $referrer->id,
                    'referred_id' => $user->id,
                    'stripe_invoice_id' => $invoice->id,
                    'commission_rate' => $rate * 100, // store as 10.00
                    'amount' => $total * $rate,
                ]);

                $createdCount++;
                $this->line("Commission created for invoice {$invoice->id}");
            }
        } while (count($invoices->data) === 100);

        $this->info("Done. Created {$createdCount} missing commissions.");
    }
}
